%% Solar Cell Experiment - First part
%
% This script is intended to display the figures corresponding to the
% measurement of a solar cell external quantum efficiency (EQE). Data is to
% be entered in the "Data Entry" section.

% Tip : Consider the use of ... to break the entry on multiple lines !

% Clearing of the work environment
clear all
close all
clc

%% Data Entry
% Wavelengths considered (in nm)
Lambda_Mes = (360:20:920);

% Power incident on the solar cell. Data between the brackets is in mW 
P_inc = 1e-6*[...
    1.9 , 5.4 , 8.4 , 16.0 , 26.3 , ...
    40.3 , 49.9 , 54.7 , 68.6 , 75.3, ...
    73.8 , 77.4 , 97.7 , 109.4 , 93.8, ...
    123.5 , 129 , 114.3 , 122.3 , 123.7, ...
    140.0 , 147.0 , 140.7 , 148.0 , 152.2,...
    142.6 , 173.8 , 121.4 , 115.9];

% Power reflected from the glass plate. Data between the brackets is in mW 
P_Refl = 1e-6*[...
    1.0 , 4.5 , 8.1 , 15.3 , 24.0 , ...
    36.0 , 43.4 , 47.0 , 58.8 , 64.3 , ...
    63.2 , 65.9 , 83.9 , 94.1 , 80.4 , ...
    106.4 , 112.2 , 99.8 , 108.7 , 109.9 , ...
    126.8 , 134.6 , 131.9 , 140.5 , 143.7 , ...
    133.5 , 161.3 , 110.6 , 102.8];

% Power reflected by the LED. Data between brackets is in mW
P_LED = 1e-6*[...
    0.0 , 0.1 , 0.1 , 0.2 , 3.1 , ...
    5.8 , 2.6 , 3.7 , 5.8 , 6.7 , ...
    7.2 , 8.6 , 11.4 , 10.8 , 6.8 , ...
    5.5 , 3.4 , 1.5 , 0.9 , 0.5 , ...
    0.3 , 0.2 , 0.1 , 0.0 , 0.0 , ...
    0.0 , 0.0 , 0.0 , 0.0];

% Short circuit current measured. Data between brackets is in mA
I_sc = -1e-3*[...
    0 , 3e-3 , 49e-3 , 132e-3 , 284e-3 , ...
    520e-3 , 729e-3 , 891e-3 , 1.2 , 1.44 , ...
    1.5 , 1.66 , 2.21 , 2.56 , 2.27 , ...
    3.07 , 3.31 , 3.00 , 3.25 , 3.30 , ...
    3.80 , 4.02 , 3.92 , 4.15 , 4.28 , ...
    4.00 , 4.80 , 3.35 , 3.14];

% Diameter of sensor aperture. Data between brackets is in [mm]
Sensor_d = 1e-3*[6];

% Diameter of the beam incident on the solar cell. Data between brackets is
% in [mm]
Spot_d = 1e-3*[88];

% Diameter of the beam reflected on the power meter head. Data between brackets is
% in [mm]
Refl_d = 1e-3*[29];

% Cells dimensions. Data between brackets is in [mm]
Cell_L = 1e-3*[62];
Cell_W = 1e-3*[31];

%% Calculation and Display of Results

% Loading the reference AM specta
Ref_Spectra = csvread('ASTMG173_DATA.csv');
Lambda_Ref = Ref_Spectra(:,1);
AM0 = Ref_Spectra(:,2);
AM_Global = Ref_Spectra(:,3);
AM_Direct = Ref_Spectra(:,4);

% Some fundamental constants
h= 6.626e-34;       
c = 299792458;
q = 1.602e-19;
kB = 1.3806e-23;

% Room temperature (in K)
T = 20+273.15;

% Corresponding areas
Sensor_A = pi*(Sensor_d/2)^2;
Refl_A = pi*(Refl_d/2)^2;
Spot_A = pi*(Spot_d/2)^2;
Cell_A = Cell_L*Cell_W;

% Wavelength (in m)
Lambda_m = Lambda_Mes*1e-9;

% Reflection coeffcient 
k = P_Refl./P_inc;

% Reflection ratio in terms of optical power (in %)
R = k*Refl_A/Spot_A*100;

% Power density measured at the solar cell 
p_Inc = P_inc/Sensor_A;
p_LED = P_LED./k/Sensor_A;

% Optical power collected by the solar cell
P_Cell = P_inc*Cell_A/Sensor_A;

% External Quantum Efficiency
EQE = I_sc/(-q)./P_Cell./Lambda_m*h*c;

% Figures
figure(1)
plot(Lambda_Mes,R,'linewidth',2)
grid on
xlabel('Wavelength [nm]')
ylabel('Reflection ratio [%]')
title('Reflection ratio of a glass plate')

figure(2)
hold on
plot(Lambda_Ref,AM_Global,'linewidth',2)
plot(Lambda_Mes,p_Inc,'r','linewidth',2)
grid on
xlabel('Wavelength [nm]')
ylabel('Power density [W/m^2/nm]')
title('Comparison between AM1.5 and halogen spectra')
legend('AM1.5 Spectrum','Halogen Spectrum','location','northwest')
xlim([Lambda_Mes(1),Lambda_Mes(end)])

figure(3)
plot(Lambda_Mes,EQE*100,'linewidth',2)
grid on
xlabel('Wavelength [nm]')
ylabel('EQE [%]')
title('External Quantum Efficiency')
